/*
 * Castaway
 *  (C) 1994 - 2002 Martin Doering, Joachim Hoenig
 *
 * $File$ - memory read/write
 *
 * This file is distributed under the GPL, version 2 or at your
 * option any later version.  See doc/license.txt for details.
 *
 */
static char     sccsid[] = "$Id: mem.c,v 1.16 2002/10/08 00:32:43 jhoenig Exp $";
#include "../config.h"

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <memory.h>

#include "../cpu/68000.h"
#include "../cpu/op68k.h"

#ifdef DEBUG
#include <assert.h>
#endif                          /* DEBUG */

// the ROM and RAM spaces
uint8           *ramBase, *romBase;

// the MMU description
static int mmuSTART;
static int mmuCONTEXT=0;

// video stuff
static int videoADDR=0;

typedef struct mmuSegment {
        uint8 SOR[2];
        uint8 SLR[2];
        } mmuSegment;
        
#define MMU_CONTEXT_NUM 4
#define MMU_SEGMENT_BY_CONTEXT 128

mmuSegment mmu[MMU_CONTEXT_NUM][MMU_SEGMENT_BY_CONTEXT];

#define MEM_IS_ROM 0
#define MEM_IS_RAM_RW 1
#define MEM_IS_RAM_RO 2
#define MEM_IS_IO 3
#define MEM_IS_BUS_ERROR 4
#define MEM_IS_SOR 5
#define MEM_IS_SLR 6

#define IO_READ 0
#define IO_WRITE 1
#define IO_BYTE 0
#define IO_WORD 1


/*
 * Read/Write memory
 */
#ifdef LITTLE_ENDIAN        /* is LITTLE_ENDIAN */
#define ReadB(address) *((int8 *) (address))
#define ReadW(addr) ((*(uint16 *)(addr) << 8) | (*(uint16 *)(addr) >> 8))
#define ReadL(address) ((uint16) ReadW(address) << 16) | (uint16) ReadW((address) + 2)
#define WriteB(address,value) *((int8 *) (address)) = (value)
#define WriteW(addr,value) *((int16 *)(addr)) = ((((uint16)(value)) << 8) | (((uint16)(value)) >> 8))
#define WriteL(address,value) WriteW((address) + 2, value); WriteW(address, (value) >> 16)
#else                           /* not LITTLE_ENDIAN */
#define ReadB(address) *((int8 *) (address))
#define ReadW(address) *((int16 *) (address))
#define ReadL(address) (*((int16 *) (address)) << 16 | *((uint16 *) (address) + 1))
#define WriteB(address,value) *((int8 *) (address)) = (value)
#define WriteW(address,value) *((int16 *) (address)) = (value)
#define WriteL(address,value) *((int16 *) (address)) = (value) >> 16; *((int16 *) (address) + 1) = (int16) (value);
#endif

/*
 * memory access jump tables
 */
void            (*mem_set_b) (uint32 address, uint8 value);
void            (*mem_set_w) (uint32 address, uint16 value);
void            (*mem_set_l) (uint32 address, uint32 value);
uint8           (*mem_get_b) (uint32 address);
uint16          (*mem_get_w) (uint32 address);
uint32          (*mem_get_l) (uint32 address);

// #define _LISA_DEBUG_

#ifdef _LISA_DEBUG_
extern char *log_box;
extern char *mmu_box;
#define TRACE_MMU0(A) {strcat(log_box,A);strcat(log_box,"\n");}
#define TRACE_MMU1(A,B) {char kbuf[256];sprintf(kbuf,A,(B));strcat(log_box,kbuf);strcat(log_box,"\n");}
#define TRACE_MMU2(A,B,C) {char kbuf[256];sprintf(kbuf,A,(B),(C));strcat(log_box,kbuf);strcat(log_box,"\n");}
#define REFRESH_MMU_BOX() refresh_mmu_box()
#else
#define TRACE_MMU0(A)
#define TRACE_MMU1(A,B)
#define TRACE_MMU2(A,B,C)
#define REFRESH_MMU_BOX
#endif

/*
 * Functions for unmapped memory areas
 */
static void    UnmappedSetB (uint32 address, uint8 value)
{
    ON_WRITE(address, value);
    ON_UNMAPPED(address, value);
}

static void    UnmappedSetW (uint32 address, uint16 value)
{
    ON_WRITE(address, value);
    ON_UNMAPPED(address, value);
}

static void    UnmappedSetL (uint32 address, uint32 value)
{
    ON_WRITE(address, value);
    ON_UNMAPPED(address, value);
}

static uint8   UnmappedGetB (uint32 address)
{
    ON_UNMAPPED(address, value);
    return 0xFFFFFFFF;
}

static uint16  UnmappedGetW (uint32 address)
{
    ON_UNMAPPED(address, value);
    return 0xFFFFFFFF;
}

static uint32  UnmappedGetL (uint32 address)
{
    ON_UNMAPPED(address, value);
    return 0xFFFFFFFF;
}

/*
 * Set RAM <= 0x800 (no access from user mode)
 */
static void    LowRamSetB (uint32 address, uint8 value)
{
    ON_WRITE(address, value);
    if (!GetFC2()) {
        ExceptionGroup0(BUSERR, address, 0);
        /* NOTREACHED */
    }
    WriteB(address, value);
}

static void    LowRamSetW (uint32 address, uint16 value)
{
    ON_WRITE(address, value);
    if (!GetFC2()) {
        ExceptionGroup0(BUSERR, address, 0);
        /* NOTREACHED */
    }
    WriteW(address, value);
}

static void    LowRamSetL (uint32 address, uint32 value)
{
    ON_WRITE(address, value);
    if (!GetFC2()) {
        ExceptionGroup0(BUSERR, address, 0);
        /* NOTREACHED */
    }
    WriteL(address, value);
}

/*
 * Set/Get RAM value
 */
void    RamSetB (uint32 address, uint8 value)
{
    ON_WRITE(address, value);
    WriteB(address, value);
}

void    RamSetW (uint32 address, uint16 value)
{
    ON_WRITE(address, value);
    WriteW(address, value);
}

void    RamSetL (uint32 address, uint32 value)
{
    ON_WRITE(address, value);
    WriteL(address, value);
}

uint8   RamGetB (uint32 address)
{
    return ReadB(address);
}

uint16  RamGetW (uint32 address)
{
    return ReadW(address);
}

uint32  RamGetL(uint32 address)
{
    return ReadL(address);
}

/*
 * Get ROM value
 */
static uint8    RomGetB (uint32 address)
{
    return ReadB(address);
}

static uint16   RomGetW (uint32 address)
{
    return ReadW(address);
}

static uint32   RomGetL (unsigned long address)
{
    return ReadL(address);
}

/*
 * Get cartridge value
 */
static uint8    CarGetB(uint32 address)
{
    return ReadB(address);
}

static uint16   CarGetW(uint32 address)
{
    return ReadW((uint16 *)(address));
}

static uint32   CarGetL(uint32 address)
{
    return ReadL((uint32 *)(address));
}

/*
 * Set/Get I/O register
 */
static void    IOSetB (uint32 address, uint8 value)
{
    ON_WRITE(address, value);
    // if (DoIOWB(address, value)) ExceptionGroup0(BUSERR, address, 0);
}

static void    IOSetW (uint32 address, uint16 value)
{
    ON_WRITE(address, value);
    // if (DoIOWW(address, value)) ExceptionGroup0(BUSERR, address, 0);
}

static void    IOSetL (uint32 address, uint32 value)
{
    ON_WRITE(address, value);
    // if (DoIOWL(address, value)) ExceptionGroup0(BUSERR, address, 0);
}

static uint8    IOGetB (uint32 address)
{
    uint8 ret;
    // if (DoIORB(address, &ret)) ExceptionGroup0(BUSERR, address, 1);
    return ret;
}

static uint16   IOGetW (uint32 address)
{
    uint16 ret;
    // if (DoIORW(address, &ret)) ExceptionGroup0(BUSERR, address, 1);
    return ret;
}

static uint32   IOGetL (uint32 address)
{
    uint32 ret;
    // if (DoIORL(address, &ret)) ExceptionGroup0(BUSERR, address, 1);
    return ret;
}

#ifdef _LISA_DEBUG_
void refresh_mmu_box(void)
{
     char buf[256];
     int context,seg;
     mmu_box[0]='\0';
     sprintf(buf,"MMU START=%d CTX=%d\n",mmuSTART,mmuCONTEXT);
     strcat(mmu_box,buf);  
     for (int context=0;context<4;context++)
         for (int seg=0;seg<128;seg++)
         {
             sprintf(buf,"[%d][%02x] or%03x l%02x t%01x\n",
                                     context,
                                     seg,
                                     ((mmu[context][seg].SOR[0])<<8|mmu[context][seg].SOR[1])&0xFFF,
                                     (mmu[context][seg].SLR[1]),
                                     (mmu[context][seg].SLR[0]&0x0F));
             strcat(mmu_box,buf);  
         }                           
 }
#endif

// define a 64Kb adress space for IO
#define IO_SPACE_MAX 0x10000
void (* ioTable[IO_SPACE_MAX]) (int *ioval,int rwMode,int size);



void accessIo(uint32 address,int *ioval,int rwMode,int size)
{
     if (ioTable[address&0xFFFF]!=NULL)
        ioTable[address&0xFFFF](ioval,rwMode,size);
     else
          TRACE_MMU1("unexpected io %06X",address);
}

void initIo(void) 
{
     int i;
     for (i=0;i<IO_SPACE_MAX;i++)
         ioTable[i]=NULL;
}

void registerIoFunc(uint32 address,void (*func)(int*,int,int))
{
     ioTable[address]=func;
 }


void setSTART(int* val,int mode,int size)
{
     mmuSTART=1; 
}
void resetSTART(int* val,int mode,int size)
{
     mmuSTART=0; 
}

void setSEG1(int* val,int mode,int size)
{
     mmuCONTEXT|=1;
}
void resetSEG1(int* val,int mode,int size)
{
     mmuCONTEXT&=2;
}
void setSEG2(int* val,int mode,int size)
{
     mmuCONTEXT|=2;
}
void resetSEG2(int* val,int mode,int size)
{
     mmuCONTEXT&=1;
}

void setVIDEOADDR(int* val,int mode,int size)
{
     // fixme what if byte access?
     if (mode==IO_WRITE)
     {
         videoADDR=(*val)&0x3F;
     }
     else
     {
         *val=videoADDR;
     }
}

uint8* getVideoMemory(void)
{
       return (&ramBase[videoADDR<<15]);
       }

void initMMUIo(void)
{
     registerIoFunc(0xE010,&setSTART);
     registerIoFunc(0xE012,&resetSTART);
     registerIoFunc(0xE00A,&setSEG1);
     registerIoFunc(0xE008,&resetSEG1);
     registerIoFunc(0xE00E,&setSEG2);
     registerIoFunc(0xE00C,&resetSEG2);
     registerIoFunc(0xE800,&setVIDEOADDR);
}

/*
* Lisa mmu entry points
*/



int lisaMMU(uint32 address,uint8** physaddress)
{
       TRACE_MMU1("Accessed address %06X",address);
       // if START mode and bit 14 not set   
       if ((mmuSTART) && ((address & 0x4000)==0x0000))
       {
            // mask bits 15 and 3
            switch (address & 0x8008) {
                   // bit 15 not set => ROM
                   case 0x0000:
                   case 0x0008:
                        *physaddress=&romBase[address&0x3FFF];
                        TRACE_MMU0("IS ROM");
                        return MEM_IS_ROM;
                   // bit 15 set and bit 3 set => SOR
                   case 0x8008:
                        *physaddress=(uint8*)&mmu[mmuCONTEXT][(address&0xFE0000)>>17].SOR; 
                        TRACE_MMU0("IS SOR");
                        return MEM_IS_SOR;
                   // bit 15 set and bit 3 not set => SLR
                   case 0x8000:
                        *physaddress=(uint8*)&mmu[mmuCONTEXT][(address&0xFE0000)>>17].SLR; 
                        TRACE_MMU0("IS SLR");
                        return MEM_IS_SLR;
                   default:
                           return MEM_IS_BUS_ERROR;
                   } // of switch
            } // of if (mmuSTART...
       // normal MMU mapped operation
       {
          uint32 seg= (address&0xFE0000)>>17;
          uint32 pageAdr=(address&0x01FE00);
          uint32 page=pageAdr>>9;
          uint32 offset=(address&0x1FF);
          int context;
          uint32 sorg;
          uint32 slim;
          
          // if supervisor mode, stick to 0 context
          if (GetS()) context=0;
             else context=mmuCONTEXT;
          sorg=(uint32)(((((mmu[context][seg].SOR[0])<<8)|
                                   mmu[context][seg].SOR[1])
                                   &0xFFF)<<9);
          slim=(uint32)mmu[context][seg].SLR[1];
          switch (mmu[context][seg].SLR[0]&0xF) {
                 // RAM RO stack
                 case 0x4 :
                      // segment limit check (stack)
                      if (((page+slim+1)&0x100)==0x000)
                      {
                         TRACE_MMU0("IS BUS ERROR RAM RO");
                         return MEM_IS_BUS_ERROR;
                      }
                      *physaddress=&ramBase[sorg+pageAdr+offset];
                      TRACE_MMU0("IS RAM STACK RO");
                      return MEM_IS_RAM_RO;
                      break;
                 // RAM RO
                 case 0x5 :
                      // segment limit check
                      if (((page+slim)&0x100)==0x100)
                      {
                         TRACE_MMU0("IS LIMIT ERROR RAM RO");
                         return MEM_IS_BUS_ERROR;
                      }
                      *physaddress=&ramBase[sorg+pageAdr+offset];
                      TRACE_MMU0("IS RAM RO");
                      return MEM_IS_RAM_RO;
                      break;
                 // RAM RW stack
                 case 0x6 :
                      // segment limit check (stack)
                      if (((page+slim+1)&0x100)==0x000);
                      {
                         TRACE_MMU0("IS LIMIT ERROR RAM STK");
                         return MEM_IS_BUS_ERROR;
                         }
                      *physaddress=&ramBase[sorg+pageAdr+offset];
                         TRACE_MMU0("IS RAM STACK RW");
                         return MEM_IS_RAM_RW;
                      break;
                 // RAM RW
                 case 0x7 :
                      // segment limit check
                      if (((page+slim)&0x100)==0x100)
                      {
                         TRACE_MMU0("IS LIMIT ERROR RAM");
                         return MEM_IS_BUS_ERROR;
                      }
                      *physaddress=&ramBase[sorg+pageAdr+offset];
                     TRACE_MMU0("IS RAM RW");
                         return MEM_IS_RAM_RW;
                      break;
                 // IO space
                 case 0x9 : 
                     TRACE_MMU0("IS IO");
                      return MEM_IS_IO;
                      break;
                 case 0xC :
                     TRACE_MMU0("IS UNMAPPED");
                     return MEM_IS_BUS_ERROR;
                 // ROM
                 case 0xF :
                        *physaddress=&romBase[(pageAdr+offset)&0x3FFF];
                     TRACE_MMU0("IS ROM2");
                        return MEM_IS_ROM;
                 default:
                     TRACE_MMU1("IS ERROR %d",mmu[context][seg].SLR[0]&0xF);
                       return MEM_IS_BUS_ERROR;
          } // of switch
       } // of normal MMU mode
       return MEM_IS_BUS_ERROR;
}


uint8 LisaGetB(uint32 address)
{
      int ret;
      uint8* physAddress;
      int ioVal;
      TRACE_MMU1("LisaGetB %06x",address);
      ret=lisaMMU(address,&physAddress);
      switch (ret) {
             case MEM_IS_ROM :
             case MEM_IS_RAM_RW :
             case MEM_IS_RAM_RO :
             case MEM_IS_SOR :
             case MEM_IS_SLR :
                    TRACE_MMU1("got %02x",physAddress[0])
                    return physAddress[0];
             case MEM_IS_IO :
                    accessIo(address,&ioVal,IO_READ,IO_BYTE);
                    TRACE_MMU1("got %02x",ioVal)
                    return (uint8)ioVal;
             case MEM_IS_BUS_ERROR :
//                     ExceptionGroup0(BUSERR, address, 0);
                     break;
//             default :
//                     ExceptionGroup0(BUSERR, address, 0);
             }
}
uint16 LisaGetW(uint32 address)
{
      int ret;
      uint8* physAddress;
      int ioVal;
      TRACE_MMU1("LisaGetW %06x",address);
      if ((address&0x1)==0x1) 
      {
// fixme
         return 0;
      }
      ret=lisaMMU(address,&physAddress);
      switch (ret) {
             case MEM_IS_ROM :
             case MEM_IS_RAM_RW :
             case MEM_IS_RAM_RO :
             case MEM_IS_SOR :
             case MEM_IS_SLR :
                    TRACE_MMU1("got %04x",(physAddress[0]<<8)|physAddress[1])
                    return (physAddress[0]<<8)|physAddress[1];
             case MEM_IS_IO :
                    accessIo(address,&ioVal,IO_READ,IO_WORD);
                    TRACE_MMU1("got %04x",ioVal)
                    return (uint16)ioVal;
             case MEM_IS_BUS_ERROR :
   //                  ExceptionGroup0(BUSERR, address, 0);
                     break;
  //           default :
  //                   ExceptionGroup0(BUSERR, address, 0);
             }
}
uint16 LisaGetWDebug(uint32 address)
{
      int ret;
      uint8* physAddress;
      int ioVal;
      if ((address&0x1)==0x1) 
      {
// fixme
         return 0;
      }
      ret=lisaMMU(address,&physAddress);
      switch (ret) {
             case MEM_IS_ROM :
             case MEM_IS_RAM_RW :
             case MEM_IS_RAM_RO :
             case MEM_IS_SOR :
             case MEM_IS_SLR :
                    return (physAddress[0]<<8)|physAddress[1];
             case MEM_IS_IO :
                    return (uint16)0xFFFF;
             case MEM_IS_BUS_ERROR :
                    return (uint16)0xFFFF;
             default :
                    return (uint16)0xFFFF;
             }
}
void LisaSetB(uint32 address,uint8 value)
{
      int ret;
      uint8* physAddress;
      int ioVal;
      TRACE_MMU2("LisaSetB %06x=%02x",address,value);
      ret=lisaMMU(address,&physAddress);
      switch (ret) {
             case MEM_IS_ROM :
             case MEM_IS_RAM_RO :
  //                   ExceptionGroup0(BUSERR, address, 0);
                     break;
             case MEM_IS_SOR :
             case MEM_IS_SLR :
             case MEM_IS_RAM_RW :
                    physAddress[0]=value;
                    return;
             case MEM_IS_IO :
                    ioVal=value;
                    accessIo(address,&ioVal,IO_WRITE,IO_BYTE);
                    return;
             case MEM_IS_BUS_ERROR :
  //                   ExceptionGroup0(BUSERR, address, 0);
                     break;
  //           default :
  //                   ExceptionGroup0(BUSERR, address, 0);
             }
}
void LisaSetW(uint32 address,uint16 value)
{
      int ret;
      uint8* physAddress;
      int ioVal;
      TRACE_MMU2("LisaSetW %06x=%04x",address,value);

      if ((address&0x1)==0x1) 
      {
// fixme
         return; 
      }
      ret=lisaMMU(address,&physAddress);
      switch (ret) {
             case MEM_IS_ROM :
             case MEM_IS_RAM_RO :
//                     ExceptionGroup0(BUSERR, address, 0);
                     break;
             case MEM_IS_SOR :
             case MEM_IS_SLR :
             case MEM_IS_RAM_RW :
                    physAddress[0]=((value&0xFF00)>>8);
                    physAddress[1]=value&0xFF;
                    break;
             case MEM_IS_IO :
                    ioVal=value;
                    accessIo(address,&ioVal,IO_WRITE,IO_WORD);
                    return;
             case MEM_IS_BUS_ERROR :
//                     ExceptionGroup0(BUSERR, address, 0);
                     break;
//             default :
//                     ExceptionGroup0(BUSERR, address, 0);
             }
}


void    LisaSetL (uint32 address, uint32 value)
{
    ON_WRITE(address, value);
    LisaSetW(address,(value&0xFFFF0000)>>16);
    LisaSetW(address+2,value&0xFFFF);
}

uint32  LisaGetL(uint32 address)
{
    return (LisaGetW(address)<<16)|LisaGetW(address+2);
}

/*
 * MemTableSet - Add memory bank handler to memory regions
 */
void    MemTableSet(void (*setbyte)(uint32, uint8),
                void (*setword)(uint32, uint16),
                void (*setlong)(uint32, uint32 ),
                uint8  (*getbyte)(uint32),
                uint16 (*getword)(uint32),
                uint32  (*getlong)(uint32) )
{
        mem_set_b = setbyte;
        mem_set_w = setword;
        mem_set_l = setlong;
        mem_get_b = getbyte;
        mem_get_w = getword;
        mem_get_l = getlong;
}



void            MemInit(void)
{
    FILE *rom;
    int ret;
    uint8 romtmp[512*32];
    int i;
    /* Set default memory access handlers */
    MemTableSet(
        LisaSetB, LisaSetW, LisaSetL,
        LisaGetB, LisaGetW, LisaGetL);

        //  
        ramBase=(uint8*)malloc(512*4096); 
        romBase=(uint8*)malloc(512*32);
        rom=fopen("bios/booth.hi","rb");
        if (rom==NULL) exit(1);
        ret=fread(&romtmp[0],512*16,1,rom);
        fclose(rom);
        rom=fopen("bios/booth.lo","rb");
        if (rom==NULL) exit(1);
        ret=fread(&romtmp[512*16],512*16,1,rom);
        fclose(rom);
        // reorder rom
        for (i=0;i<512*16;i++)
        {
         romBase[i*2]=romtmp[i];
         romBase[i*2+1]=romtmp[i+512*16];
        }
        mmuSTART=1;
        initIo();
        initMMUIo();
}

